
/*
  This file implements CRangeList class,
  which is used to store and update information
  about memory ranges within debugging process.
  First memory range is main EXE file; other are DLLs it uses.
  Each byte within each memory range can have
  different properties, which all are described here.
*/

#define LOG_RANGE

#define R_OPCODE               0x01
#define R_MULTITHREAD          0x02
#define R_MULTIEXEC            0x04
#define R_EXCEPTION            0x08
#define R_SEHHANDLER           0x10
#define R_VARIABLE             0x20
#define R_reserved1            0x40
#define R_reserved2            0x80

struct range_struct : CListEntry
{
  DWORD  base;                  // base address
  DWORD  size;                  // image size (virtual)
  char   filename[1024];        // EXE or DLL filename
  // big arrays, for each byte in virtual memory, relative to base
  BYTE*  flag;                  // R_xxx
  BYTE*  mem0;                  // initial memory state
  BYTE*  mem1;                  // memory state after trace completed
  DWORD* thread;                // thread id (or -1 if multithread code)
  DWORD* l_index;   // local  # of instruction (or while 1st cycle iteration)
  DWORD* g_index;   // global # of instruction (--//--)
  DWORD* prev_ins;              // prev instruction rva (or -1 if label)
  DWORD* count;                 // execution count (0=never, 1=once, ...)
};

void __cdecl CRangeListOnFree(void* entry)
{
  range_struct*t = (range_struct*)entry;
  ZFreeV( (void**)&t->flag     );
  ZFreeV( (void**)&t->mem0     );
  ZFreeV( (void**)&t->mem1     );
  ZFreeV( (void**)&t->thread   );
  ZFreeV( (void**)&t->count    );
  ZFreeV( (void**)&t->l_index  );
  ZFreeV( (void**)&t->g_index  );
  ZFreeV( (void**)&t->prev_ins );
} // CRangeList::OnFree()

class CRangeList : public CList
{
  public:
  CRangeList() : CList( sizeof(range_struct) ) { OnFree = CRangeListOnFree; }
  range_struct* Find(DWORD addr);
  int Add(HANDLE handle, DWORD base);
  int Del(DWORD base);
}; // class CRangeList

range_struct* CRangeList::Find(DWORD addr)
{
  range_struct* r;
  ForEach(range_struct, r)
  if ( (addr >= r->base) && (addr < r->base + r->size) )
    return r;
  return NULL;
} // CRangeList::Find()

int CRangeList::Add(HANDLE handle, DWORD base)
{
  DWORD peptr, size;
  if (read_memory(handle, base+0x3C, (BYTE*)&peptr, 4)==0)       // 0x3C=MZ.PE_PTR
  {
#ifdef LOG_RANGE
    log("ERROR: CRangeList::Add(base=%08X): cant read MZ.PE_PTR\n", base);
#endif
    return 0;
  }
  if (read_memory(handle, base+peptr+0x50, (BYTE*)&size, 4)==0)  // 0x50=PE.ImageSize
  {
#ifdef LOG_RANGE
    log("ERROR: CRangeList::Add(base=%08X): cant read PE.ImageSize\n", base);
#endif
    return 0;
  }

  range_struct* r;
  ForEach(range_struct, r)
  if (  ((base        >= r->base) && (base        < r->base + r->size)) ||
        ((base + size >  r->base) && (base + size < r->base + r->size)) )
  {
#ifdef LOG_RANGE
    log("ERROR: CRangeList::Add(base=%08X,size=%08X): range intersection with base=%08X,size=%08X\n", base, size, r->base, r->size);
#endif
    return 0;
  }

  r = (range_struct*) Alloc();
  if (r == NULL)
  {
#ifdef LOG_RANGE
    log("ERROR: CRangeList::Add(base=%08X,size=%08X): cant allocate memory for range_struct\n", base, size);
#endif
    return 0;
  }

  r->base = base;
  r->size = size;

  r->flag     = (BYTE*) ZAlloc( size+1 );
  r->mem0     = (BYTE*) ZAlloc( size+1 );
  r->mem1     = (BYTE*) ZAlloc( size+1 );
  r->thread   = (DWORD*)ZAlloc( (size+1) * 4 );
  r->count    = (DWORD*)ZAlloc( (size+1) * 4 );
  r->l_index  = (DWORD*)ZAlloc( (size+1) * 4 );
  r->g_index  = (DWORD*)ZAlloc( (size+1) * 4 );
  r->prev_ins = (DWORD*)ZAlloc( (size+1) * 4 );

  if (  (r->flag     == NULL) ||
        (r->mem0     == NULL) ||
        (r->mem1     == NULL) ||
        (r->thread   == NULL) ||
        (r->count    == NULL) ||
        (r->l_index  == NULL) ||
        (r->g_index  == NULL) ||
        (r->prev_ins == NULL)  )
  {
#ifdef LOG_RANGE
    log("ERROR: CRangeList::Add(base=%08X,size=%08X): cant allocate memory for big arrays\n");
#endif
    return 0;
  }

  if (read_memory(handle, base, r->mem0, size)==0)
  {
#ifdef LOG_RANGE
    log("ERROR: CRangeList::Add(base=%08X,size=%08X): cant read the whole image\n", base, size);
#endif
    return 0;
  }

  DWORD exrva  = *(DWORD*)&r->mem0[ peptr+0x78 ];
  DWORD exsize = *(DWORD*)&r->mem0[ peptr+0x7C ];
  if ( (exrva  != NULL) &&
       (exsize != 0   ) &&
       (exrva  < size ) &&
       (exrva+exsize < size))
  {
    DWORD namerva = *(DWORD*)&r->mem0[ exrva+0x0C ];
    if (namerva < size)
    {
      if (read_memory(handle, base+namerva, (BYTE*)&r->filename[0], 260)==0)
      {
#ifdef LOG_RANGE
        log("ERROR: CRangeList::Add(base=%08X,size=%08X): cant read module name\n", base, size);
#endif
        return 0;
      }
    }
  }

#ifdef LOG_RANGE
  log("new range: base=%08X size=%08X end=%08X name=[%s]\n", r->base, r->size, r->base+r->size, r->filename);
#endif

  Attach(r);

  return 1;
} // CRangeList::Add()

int CRangeList::Del(DWORD base)
{
  range_struct* r = Find(base);
  if (r == NULL)
  {
#ifdef LOG_RANGE
    log("ERROR: CRangeList::Del(base=%08X): base not found in rangelist\n", base);
#endif
    return 0;
  }

  Detach( (void*)r );
  Free  ( (void*)r );

  return 1;
} // CRangeList::Del()
